
;*****************************************************************************
;
;             Sample I2C Single Master Routines for the 87LPC764
;              (taken from AN422 - 8XC751 as I2C Bus Master)
;
;*****************************************************************************

$TITLE(87LPC764 Single Master I2C Routines)
$INCLUDE (REG764.INC)


; Value definitions.

CTVAL      EQU     02h                 ;CT1, CT0 bit values for I2C.


; Masks for I2CFG bits.

BTIR       EQU     10h                 ;Mask for TIRUN bit.
BMRQ       EQU     40h                 ;Mask for MASTRQ bit.


; Masks for I2CON bits.

BCXA       EQU     80h                 ;Mask for CXA bit.
BIDLE      EQU     40h                 ;Mask for IDLE bit.
BCDR       EQU     20h                 ;Mask for CDR bit.
BCARL      EQU     10h                 ;Mask for CARL bit.
BCSTR      EQU     08h                 ;Mask for CSTR bit.
BCSTP      EQU     04h                 ;Mask for CSTP bit.
BXSTR      EQU     02h                 ;Mask for XSTR bit.
BXSTP      EQU     01h                 ;Mask for XSTP bit.


; RAM locations used by I2C routines.

BitCnt     DATA    21h                 ;I2C bit counter.
ByteCnt    DATA    22h
SlvAdr     DATA    23h                 ;Address of active slave.
SubAdr     DATA    24h

RcvDat     DATA    25h                 ;I2C receive data buffer (4 bytes).
                                       ;  addresses 25h through 28h.

XmtDat     DATA    29h                 ;I2C transmit data buffer (4 bytes).
                                       ;  addresses 29h through 2Ch.

StackSave  DATA    2Dh                 ;Saves stack address for bus recovery.

Flags      DATA    20h                 ;I2C software status flags.
NoAck      BIT     Flags.0             ;Indicates missing acknowledge.
Fault      BIT     Flags.1             ;Indicates a bus fault of some kind.
Retry      BIT     Flags.2             ;Indicates that last I2C transmission
                                       ;  failed and should be repeated.

SCL        BIT     P1.2                ;Port bit for I2C serial clock line.
SDA        BIT     P1.3                ;Port bit for I2C serial data line.

;*****************************************************************************
;                                Begin Code
;*****************************************************************************

; Reset and interrupt vectors.

           AJMP    Reset               ;Reset vector at address 0.


; A timer I timeout usually indicates a 'hung' bus.

           ORG     73h                 ;Timer I (I2C timeout) interrupt.
TimerI:    SETB    CLRTI               ;Clear timer I interrupt.
           CLR     TIRUN
           ACALL   ClrInt              ;Clear interrupt pending.
           MOV     SP,StackSave        ;Restore stack for return to main.
           AJMP    Recover             ;Attempt bus recovery.
ClrInt:    RETI


;*****************************************************************************
;                    Main Transmit and Receive Routines
;*****************************************************************************

; Send data byte(s) to slave.
;   Enter with slave address in SlvAdr, data in XmtDat buffer, # of data 
;   bytes to send in ByteCnt.

SendData:  CLR     NoAck               ;Clear error flags.
           CLR     Fault
           CLR     Retry
           MOV     StackSave,SP        ;Save stack address for bus fault.
           MOV     A,SlvAdr            ;Get slave address.
           ACALL   SendAddr            ;Get bus and send slave address.
           JB      NoAck,SDEX          ;Check for missing acknowledge.
           JB      Fault,SDatErr       ;Check for bus fault.
           MOV     R0,#XmtDat          ;Set start of transmit buffer.

SDLoop:    MOV     A,@R0               ;Get data for slave.
           INC     R0
           ACALL   XmitByte            ;Send data to slave.
           JB      NoAck,SDEX          ;Check for missing acknowledge.
           JB      Fault,SDatErr       ;Check for bus fault.
           DJNZ    ByteCnt,SDLoop

SDEX:      ACALL   SendStop            ;Send an I2C stop.
           RET


; Handle a transmit bus fault.

SDatErr:   AJMP    Recover             ;Attempt bus recovery.


; Receive data byte(s) from slave.
;   Enter with slave address in SlvAdr, # of data bytes requested in ByteCnt.
;   Data returned in RcvDat buffer.

RcvData:   CLR     NoAck               ;Clear error flags.
           CLR     Fault
           CLR     Retry
           MOV     StackSave,SP        ;Save stack address for bus fault.
           MOV     A,SlvAdr            ;Get slave address.
           SETB    ACC.0               ;Aet bus read bit.
           ACALL   SendAddr            ;Send slave address.
           JB      NoAck,RDEX          ;Check for missing acknowledge.
           JB      Fault,RDatErr       ;Check for bus fault.

           MOV     R0,#RcvDat          ;Set start of receive buffer.
           DJNZ    ByteCnt,RDLoop      ;Check for count = 1 byte only.
           SJMP    RDLast

RDLoop:    ACALL   RDAck               ;Get data and send an acknowledge.
           JB      Fault,RDatErr       ;Check for bus fault.
           MOV     @R0,A               ;Save data.
           INC     R0
           DJNZ    ByteCnt,RDLoop      ;Repeat until last byte.

RDLast:    ACALL   RcvByte             ;Get last data byte from slave.
           JB      Fault,RDatErr       ;Check for bus fault.
           MOV     @R0,A               ;Save data.

           MOV     I2DAT,#80h          ;Send negative acknowledge.
           JNB     ATN,$               ;Wait for NAK sent.
           JNB     DRDY,RDatErr        ;Check for bus fault.

RDEX:      ACALL   SendStop            ;Send an I2C bus stop.
           RET


; Handle a receive bus fault.

RDatErr:   AJMP    Recover             ;Attempt bus recovery.


; Send data byte(s) to slave with subaddress.
;   Enter with slave address in ACC, subaddress in SubAdr, # of bytes to
;     send in ByteCnt, data in XmtDat buffer.

SendSub:   CLR     NoAck               ;Clear error flags.
           CLR     Fault
           CLR     Retry
           MOV     StackSave,SP        ;Save stack address for bus fault.
           MOV     A,SlvAdr            ;Get slave address.
           ACALL   SendAddr            ;Get bus and send slave address.
           JB      NoAck,SSEX          ;Check for missing acknowledge.
           JB      Fault,SSubErr       ;Check for bus fault.

           MOV     A,SubAdr            ;Get slave subaddress.
           ACALL   XmitByte            ;Send subaddress.
           JB      NoAck,SSEX          ;Check for missing acknowledge.
           JB      Fault,SSubErr       ;Check for bus fault.
           MOV     R0,#XmtDat          ;Set start of transmit buffer.

SSLoop:    MOV     A,@R0               ;Get data for slave.
           INC     R0
           ACALL   XmitByte            ;Send data to slave.
           JB      NoAck,SSEX          ;Check for missing acknowledge.
           JB      Fault,SSubErr       ;Check for bus fault.
           DJNZ    ByteCnt,SSLoop

SSEX:      ACALL   SendStop            ;Send an I2C stop.
           RET


; Handle a transmit bus fault.

SSubErr:   AJMP    Recover             ;Attempt bus recovery.


; Receive data byte(s) from slave with subaddress.
;   Enter with slave address in SlvAdr, subaddress in SubAdr, # of data bytes
;     requested in ByteCnt. Data returned in RcvDat buffer.

RcvSub:    CLR     NoAck               ;Clear error flags.
           CLR     Fault
           CLR     Retry
           MOV     StackSave,SP        ;Save stack address for bus fault.
           MOV     A,SlvAdr            ;Get slave address.
           ACALL   SendAddr            ;Send slave address.
           JB      NoAck,RSEX          ;Check for missing acknowledge.
           JB      Fault,RSubErr       ;Check for bus fault.

           MOV     A,SubAdr            ;Get slave subaddress.
           ACALL   XmitByte            ;Send subaddress.
           JB      NoAck,RSEX          ;Check for missing acknowledge.
           JB      Fault,RSubErr       ;Check for bus fault.

           ACALL   RepStart            ;Send repeated start.
           JB      Fault,RSubErr       ;Check for bus fault.
           MOV     A,SlvAdr            ;Get slave address.
           SETB    ACC.0               ;Set bus read bit.
           ACALL   SendAd2             ;Send slave address.
           JB      NoAck,RSEX          ;Check for missing acknowledge.
           JB      Fault,RSubErr       ;Check for bus fault.

           MOV     R0,#RcvDat          ;Set start of receive buffer.
           DJNZ    ByteCnt,RSLoop      ;Check for count = 1 byte only.
           SJMP    RSLast

RSLoop:    ACALL   RDAck               ;Get data and send an acknowledge.
           JB      Fault,RSubErr       ;Check for bus fault.
           MOV     @R0,A               ;Save data.
           INC     R0
           DJNZ    ByteCnt,RSLoop      ;Repeat until last byte.

RSLast:    ACALL   RcvByte             ;Get last data byte from slave.
           JB      Fault,RSubErr       ;Check for bus fault.
           MOV     @R0,A               ;Save data.

           MOV     I2DAT,#80h          ;Send negative acknowledge.
           JNB     ATN,$               ;Wait for NAK sent.
           JNB     DRDY,RSubErr        ;Check for bus fault.

RSEX:      ACALL   SendStop            ;Send an I2C bus stop.
           RET


; Handle a receive bus fault.

RSubErr:   AJMP    Recover             ;Attempt bus recovery.


;*****************************************************************************
;                               Subroutines
;*****************************************************************************

; Send address byte.
;   Enter with address in ACC.

SendAddr:  MOV     I2CFG,#BMRQ+BTIR+CTVAL ;Request I2C bus.
           JNB     ATN,$               ;Wait for bus granted.
           JNB     Master,SAErr        ;Should have become the bus master.
SendAd2:   MOV     I2DAT,A             ;Send first bit, clears DRDY.
           MOV     I2CON,#BCARL+BCSTR+BCSTP ;Clear start, releases SCL.
           ACALL   XmitAddr            ;Finish sending address.
           RET

SAErr:     SETB    Fault               ;Return bus fault status.
           RET


; Byte transmit routine.
;   Enter with data in ACC.
;   XmitByte : transmits 8 bits.
;   XmitAddr : transmits 7 bits (for address only).

XmitAddr:  MOV     BitCnt,#8           ;Set 7 bits of address count.
           SJMP    XmBit2

XmitByte:  MOV     BitCnt,#8           ;Set 8 bits of data count.
XmBit:     MOV     I2DAT,A             ;Send this bit.
XmBit2:    RL      A                   ;Get next bit.
           JNB     ATN,$               ;Wait for bit sent.
           JNB     DRDY,XMErr          ;Should be data ready.
           DJNZ    BitCnt,XmBit        ;Repeat until all bits sent.
           MOV     I2CON,#BCDR+BCXA    ;Switch to receive mode.
           JNB     ATN,$               ;Wait for acknowledge bit.
           JNB     RDAT,XMBX           ;Was there an ack?
           SETB    NoAck               ;Return no acknowledge status.
XMBX:      RET

XMErr:     SETB    Fault               ;Return bus fault status.
           RET


; Byte receive routines.
;   RDAck   : receives a byte of data, then sends an acknowledge.
;   RcvByte : receives a byte of data.
;   Data returned in ACC.

RDAck:     ACALL   RcvByte             ;Receive a data byte.
           MOV     I2DAT,#0            ;Send receive acknowledge.
           JNB     ATN,$               ;Wait for acknowledge sent.
           JNB     DRDY,RdErr          ;Check for bus fault.
           RET

RcvByte:   MOV     BitCnt,#8           ;Set bit count.
           CLR     A                   ;Init received byte to 0.
RBit:      ORL     A,I2DAT             ;Get bit, clear ATN.
           RL      A                   ;Shift data.
           JNB     ATN,$               ;Wait for next bit.
           JNB     DRDY,RdErr          ;Should be data ready.
           DJNZ    BitCnt,RBit         ;Repeat until 7 bits are in.
           MOV     C,RDAT              ;Get last bit, don't clear ATN.
           RLC     A                   ;Form full data byte.
           RET

RdErr:     SETB    Fault               ;Return bus fault status.
           RET


; I2C stop routine.

SendStop:  CLR     MASTRQ              ;Release bus mastership.
           MOV     I2CON,#BCDR+BXSTP   ;Generate a bus stop.
           JNB     ATN,$               ;Wait for atn.
           MOV     I2CON,#BCDR         ;Clear data ready.
           JNB     ATN,$               ;Wait for stop sent.
           MOV     I2CON,#BCARL+BCSTP+BCXA ;Clear I2C bus.
           CLR     TIRUN               ;Stop timer I.
           RET


; I2C repeated start routine.
;   Enter with address in ACC.

RepStart:  MOV     I2CON,#BCDR+BXSTR   ;Send repeated start.
           JNB     ATN,$               ;Wait for ATN.
           MOV     I2CON,#BCDR         ;Clear data ready.
           JNB     ATN,$               ;Wait for repeated start sent.
           MOV     I2CON,#BCARL+BCSTR  ;Clear start.
           RET


; Bus fault recovery routine.

Recover:   ACALL   FixBus              ;See if bus is dead or can be 'fixed'.
           JC      BusReset            ;If not 'fixed', try extreme measures.
           SETB    Retry               ;If bus OK, return to main routine.
           CLR     Fault
           CLR     NoAck
           SETB    CLRTI
           SETB    TIRUN               ;Enable timer I.
           SETB    ETI                 ;Turn on timer I interrupts.
           RET

; This routine tries a more extreme method of bus recovery.
;   This is used if SCL or SDA are stuck and cannot otherwise be freed.
;   (will return to the Recover routine when Timer I times out)

BusReset:  CLR     MASTRQ              ;Release bus.
           MOV     I2CON,#0BCh         ;Clear all I2C flags.
           SETB    TIRUN
           SJMP    $                   ;Wait for timer I timeout (this will
                                       ;  reset the I2C hardware).


; This routine attempts to regain control of the I2C bus after a bus fault.
;   Returns carry clear if successful, carry set if failed.

FixBus:    CLR     MastRQ              ;Turn off I2C functions.
           SETB    C
           SETB    SCL                 ;Insure I/O port is not locking I2C.
           SETB    SDA
           JNB     SCL,FixBusEx        ;If SCL is low, bus cannot be 'fixed'.
           JB      SDA,RStop           ;If SCL & SDA are high, force a stop.
           MOV     BitCnt,#9           ;Set max # of tries to clear bus.
ChekLoop:  CLR     SCL                 ;Force an I2C clock.
           ACALL   SDelay
           JB      SDA,RStop           ;Did it work?
           SETB    SCL
           ACALL   SDelay
           DJNZ    BitCnt,ChekLoop     ;Repeat clocks until either SDA clears
                                       ;  or we run out of tries.
           SJMP    FixBusEx            ;Failed to fix bus by this method.

RStop:     CLR     SDA                 ;Try forcing a stop since SCL & SDA
           ACALL   SDelay              ;  are both high.
           SETB    SCL
           ACALL   SDelay
           SETB    SDA
           ACALL   SDelay
           JNB     SCL,FixBusEx        ;Are SCL & SDA still high? If so,
           JNB     SDA,FixBusEx        ;  assume bus is now OK, and return
           CLR     C                   ;  with carry cleared.
FixBusEx:  RET


; Short delay routine (10 machine cycles).

SDelay:    NOP
           NOP
           NOP
           NOP
           NOP
           NOP
           NOP
           NOP
           RET


;*****************************************************************************
;                               Main Program
;*****************************************************************************

Reset:     MOV     SP,#07h             ;Set stack location.
           SETB    ETI                 ;Enable timer I interrupts.
           SETB    EA                  ;Enable global interrupts.
           MOV     XmtDat,#11          ;Set up transmit data.
           MOV     XmtDat+1,#22        ;Set up transmit data.
           MOV     XmtDat+2,#44        ;Set up transmit data.
           MOV     XmtDat+3,#88        ;Set up transmit data.
           MOV     RcvDat,#0           ;Clear receive data.
           MOV     RcvDat+1,#0         ;Clear receive data.
           MOV     RcvDat+2,#0         ;Clear receive data.
           MOV     RcvDat+3,#0         ;Clear receive data.

MainLoop:  MOV     SlvAdr,#48h         ;Set slave address (8-bit I/O port).
           MOV     ByteCnt,#1          ;Set up byte count.
           ACALL   SendData            ;Send data to slave.
           JB      Retry,MainLoop

ML2:       MOV     ByteCnt,#1          ;Set up byte count.
           ACALL   RcvData             ;Read data from slave.
           JB      Retry,ML2

SL1:       MOV     SlvAdr,#0A0h        ;Set slave address (RAM chip).
           MOV     SubAdr,#0h          ;Set slave subaddress.
           MOV     ByteCnt,#4          ;Set up byte count.
           ACALL   SendSub
           JB      Retry,SL1

SL2:       MOV     ByteCnt,#4          ;Set up byte count.
           ACALL   RcvSub
           JB      Retry,SL2

           INC     XmtDat
           INC     XmtDat+1
           INC     XmtDat+2
           INC     XmtDat+3
           SJMP    MainLoop            ;Do it all again.

           END
