#include "reg591.h"
#include "i2ceeprom.h"

#define EEPROM_ADDRESS 0xA8                   // I2C address of EEPROM
#define TIMEOUT 0xAA0F                        // timer reload value gives 22ms timeout period
                                              // for waiting on SLA + W ack

void init_i2c(void);
void eeprom_write_byte(unsigned int address, unsigned char byte);
unsigned char eeprom_read_byte(unsigned int address);
void i2c_isr(void);
void start_timer(void);
void stop_timer(void);

enum i2c_states { LOWER_ADDRESS, DATA_BYTE, FINISHED, DUMMY_WRITE }; // states during an I2C operation
enum i2c_operations { READ, WRITE };          // possible I2C operations are reading or writing

unsigned int  i2c_storage_address;            // holds the address to read from or write to
unsigned char i2c_byte;                       // holds the value read or the value to write
unsigned char i2c_state;                      // holds the status - LOWER_ADDRESS, DATA BYTE, etc.
unsigned char i2c_status;                     // holds the state - I2C_ERROR, I2C_OK or I2C_BUSY
unsigned char i2c_operation;                  // holds the operation - READ or WRITE

// function init_i2c
// initializes the I2C peripheral
// nothing is passed. nothing is returned
void init_i2c(void)
{
  P1M1 |= 0xC0;                               // set P1.6 and P1.7 open drain
  P1M2 |= 0xC0;
  TMOD |= 0x01;                               // timer 0 mode 1
  ES1 = 1;                                    // enable SIO1 interrupts
  EA = 1;                                     // global interrupt enable
}

// function eeprom_write_byte
// stores a byte in the EEPROM at a specific address
// passed is the 12-bit address and the byte to store
// nothing is returned.
// the result of the operation may be read in i2c_status and will be
// I2C_ERROR or I2C_OK
void eeprom_write_byte(unsigned int address, unsigned char byte)
{
  i2c_storage_address = address;              // copy the address
  i2c_byte = byte;                            // copy the byte to store
  i2c_status = I2C_BUSY;                      // I2C is now busy
  i2c_operation = WRITE;                      // a write operation is being performed
  S1CON = 0x40;                               // enable I2C - no acknowledgements will be generated
  STA = 1;                                    // request to transmit a start condition - places I2C peripheral in
                                              // master transmitter mode
  while(i2c_status == I2C_BUSY);              // wait until I2C is no longer busy
}

// function eeprom_read_byte
// reads a byte at a specific address in the EEPROM
// passed is the 12-bit address
// returned is the byte if the operation was successful.
// the result of the operation may be read in i2c_status and will
// be I2C_ERROR or I2C_OK
unsigned char eeprom_read_byte(unsigned int address)
{
  i2c_storage_address = address;               // copy the address
  i2c_status = I2C_BUSY;                       // I2C is now busy
  i2c_operation = READ;                        // read operation is being performed
  i2c_state = DUMMY_WRITE;                     // initially a dummy write is to be performed
  S1CON = 0x40;                                // enable I2C - no acknowledgements will be generated
  STA = 1;                                     // request to transmit a start condition - this will place the
                                               // I2C peripheral in master transmitter mode
  while(i2c_status == I2C_BUSY);               // wait until I2C is no longer busy
  if (i2c_status == I2C_OK)
    return i2c_byte;                           // if operation was successful then return read value
  else
    return 0xFF;                               // return a constant value if there was an error
}

// function start_timer
// initializes and starts timer 0 to implement a timeout period
// nothing is passed. nothing is returned
void start_timer(void)
{
  TL0 = TIMEOUT & 0xFF;                       // load timer reload value
  TH0 = (TIMEOUT >> 8) & 0xFF;
  TR0 = 1;                                    // start timer 0
}


// function stop_timer
// stops timer 0 and clears the timer overflow flag
// nothing is passed. nothing is returned
void stop_timer(void)
{
  TR0 = 0;                                    // stop timer 0
  TF0 = 0;                                    // clear timer 0 overflow flag
}

// function i2c_isr
// I2C interrupt service routine. An interrupt is requested whenever an event occurs
// on the I2C bus causing the I2C peripheral to change state
// The state is held in the S1STA SFR
// nothing is passed. nothing is returned.
// note: if debugging printfs are inserted into this function the execution
// will be slowed down to the point where the timeout occurs.
void i2c_isr(void) interrupt 5 using 1
{
  switch(S1STA)
  {
    // START CONDITION TRANSMITTED
    case 0x08:
      STA = 0;                                // clear start condition flag so another start condition is not requested
	  S1DAT = EEPROM_ADDRESS & 0xFE;          // transmit EEPROM address + write command (SLA + W)
	  start_timer();                          // start timer 0 to start measuring the timeout period
      break;
	// REPEATED START CONDITION TRANSMITTED
    case 0x10:
      STA = 0;                                // clear start condition flag so another start condition is not requested
      if (i2c_operation == READ && i2c_state != DUMMY_WRITE) // if performing a read operation and dummy write is completed then...
	  {
	    S1DAT = EEPROM_ADDRESS | 0x01;        // transmit EEPROM address + read command (SLA + R)
  	  }
	  else
	  {
        S1DAT = EEPROM_ADDRESS & 0xFE;        // transmit EEPROM address + write command (SLA + W)
		                                      // this situation will arise if no resonse was received from the EEPROM
									          // and another attempt is being made
      }
      break;
	// SLAVE ADDRESS + WRITE TRANSMITTED - ACK RECEIVED
    case 0x18:
	  stop_timer();                           // stop timer 0 - a response was received within the timeout period
      S1DAT = (i2c_storage_address >> 8) & 0x0F; // transmit upper 4 bits of address
	  i2c_state = LOWER_ADDRESS;              // next byte to be transmitted is lower part of address
      break;
	// SLAVE ADDRESS + WRITE TRANSMITTED - NO ACK RECEIVED
    case 0x20:
	  if (TF0)                                // if timer 0 overflowed then timeout period reached. no response.
	  {
	    STO = 1;                              // request to transmit a stop condition
		stop_timer();                         // stop timer 0
		i2c_status = I2C_ERROR;               // status of operation is ERROR
	  }
	  else
	  {
        STA = 1;                              // request to transmit a repeated start - try again to communicate
	  }
      break;
	// DATA BYTE OR UPPER ADDRESS OR LOWER ADDRESS TRANSMITTED - ACK RECEIVED
    case 0x28:
	  switch(i2c_state)
	  {
	    case LOWER_ADDRESS:                   // next byte to transmit is lower part of address
	      S1DAT = i2c_storage_address & 0xFF; // transmit lower 8 bits of address
	      i2c_state = DATA_BYTE;              // next time around transmit the data byte
	      break;
	    case DATA_BYTE:                       // next byte to transmit is the data byte
          if (i2c_operation == READ)          // if performing a read then the dummy write cycle
		                                      // ends here. no data byte is transmitted
		  {
		    STA = 1;                          // request to transmit repeated start
		  }
		  else
		  {
            S1DAT = i2c_byte;                 // transmit the data byte
		    i2c_state = FINISHED;             // next time around the write cycle is finished
          }
          break;
		// finished writing byte
	    case FINISHED:                        // data byte was successfully transmitted
		  STO = 1;                            // request to transmit a stop condition
		  i2c_status = I2C_OK;                // status of operation is OK
		  break;
      }
	  break;
	// DATA BYTE OR UPPER ADDRESS OR LOWER ADDRESS TRANSMITTED - NO ACK RECEIVED
	case 0x30:
	  STO = 1;                                // no response. request to transmit a stop condition
	  i2c_status = I2C_ERROR;                 // status of operation is ERROR
      break;
	// SLAVE ADDRESS + READ TRANSMITTED - ACK RECEIVED
	case 0x40:
	  break;                                  // no action is performed. next time an interrupt is
	                                          // generated the data byte will have been received from
											  // the EEPROM
	// SLAVE ADDRESS + READ TRANSMITTED - NO ACK RECEIVED
	case 0x48:
      STO = 1;                                // request to transmit stop condition
	  i2c_status = I2C_ERROR;                 // status of operation is ERROR
      break;
	// DATA BYTE RECEIVED - NO ACK TRANSMITTED
    case 0x58:
      i2c_byte = S1DAT;                       // data byte has been received and no acknowledge was
	                                          // transmitted as per the datasheet. read the data byte
	  i2c_status = I2C_OK;                    // status of operation is OK
      STO = 1;                                // request to transmit stop condition
 	  break;
	// UNKNOWN STATE
    default:
	  STO = 1;                                // request to transmit stop condition
	  i2c_status = I2C_ERROR;                 // status of operation is ERROR
	  ES1 = 0;                                // disable the I2C interrupt
	  break;
  }
  SI = 0;                                     // clear the I2C interrupt flag. This causes the next event on the
                                              // I2C bus to be performed, possibly resulting in the request of
											  // another I2C interrupt
}

